const db = require("../config/db");
const redis = require("../config/redis");
const bcrypt = require("bcryptjs");

class Vendor {
  // 🟢 Create Clients Table
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS clients (
        client_id SERIAL PRIMARY KEY,
        client_name VARCHAR(500) NOT NULL UNIQUE,
        email VARCHAR(500) NOT NULL UNIQUE,
        telephone VARCHAR (50) NOT NULL UNIQUE,
        address TEXT,
        street TEXT,
        city VARCHAR(500),
        zipcode VARCHAR(20),
        region VARCHAR(500),
        country VARCHAR(20),
        client_status VARCHAR(50),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `;

    try {
      await db.query(query);
      console.log("✅ clients table is ready.");
    } catch (err) {
      console.error("❌ Error creating clients table:", err);
    }
  }

  static async create({
    client_name,
    email,
    telephone,
    password,
    address,
    street,
    city,
    zipcode,
    region,
    country,
    client_status,
    role = "client",
    profile_img, // New field
  }) {
    const hashedPassword = await bcrypt.hash(password, 10); // Hash password

    let query = `
      INSERT INTO clients 
      (client_name, email, telephone, password, address, street, city, zipcode, region, country, client_status, role`;
    let values = [
      client_name,
      email,
      telephone,
      hashedPassword,
      JSON.stringify(address),
      street,
      city,
      zipcode,
      region,
      country,
      client_status,
      role,
    ];

    if (profile_img) {
      query += `, profile_img`;
      values.push(profile_img);
    }

    query += `) VALUES (${values.map((_, i) => `$${i + 1}`).join(", ")}) 
      RETURNING client_id`;

    const { rows } = await db.query(query, values);

    return rows[0].client_id;
  }

  static async getAll() {
    const cacheKey = "clients:all";
    const cachedClients = await redis.get(cacheKey);

    if (cachedClients) {
      return JSON.parse(cachedClients);
    }

    const query = `
    SELECT 
        c.client_id, 
        c.client_name, 
        c.email, 
        c.telephone, 
        c.street, 
        c.city, 
        c.zipcode, 
        r.region_name,  -- Get the region name
        co.country_name, -- Get the country name
        c.client_status, 
        c.created_at, 
        c.role, 
        c.profile_img, 
        c.address
    FROM clients c
    LEFT JOIN regions r ON c.region = r.region_id::VARCHAR  -- Cast INTEGER to VARCHAR
    LEFT JOIN countries co ON c.country = co.country_id::VARCHAR; -- Cast INTEGER to VARCHAR
`;

    try {
      const { rows } = await db.query(query);
      const transformedClients = rows.map((client) => {
        return {
          client_id: client.client_id,
          client_name: client.client_name,
          email: client.email,
          telephone: client.telephone,
          street: client.street,
          city: client.city,
          zipcode: client.zipcode,
          region: client.region_name || "Unknown", // Get region name
          country: client.country_name || "Unknown", // Get country name
          client_status: client.client_status === "1" ? "Active" : "Inactive",
          created_at: client.created_at,
          role: client.role,
          profile_img: client.profile_img ? client.profile_img : null,
          address: client.address ? JSON.parse(client.address) : null,
        };
      });

      await redis.setex(cacheKey, 3600, JSON.stringify(transformedClients));
      return transformedClients;
    } catch (err) {
      console.error("❌ Error executing getAll query:", err);
      throw err;
    }
  }

  static async getById(client_id) {
    const cacheKey = `clients:${client_id}`;
    const cachedClient = await redis.get(cacheKey);

    if (cachedClient) {
      return JSON.parse(cachedClient);
    }

    const query = `
      SELECT 
        c.client_id, 
        c.client_name, 
        c.email, 
        c.telephone, 
        c.street, 
        c.city, 
        c.zipcode, 
        r.region_name,  -- Get the region name
        co.country_name, -- Get the country name
        c.client_status, 
        c.created_at, 
        c.role, 
        c.profile_img, 
        c.address
        FROM clients c
        LEFT JOIN regions r ON c.region = r.region_id::VARCHAR  -- Cast INTEGER to VARCHAR
        LEFT JOIN countries co ON c.country = co.country_id::VARCHAR -- Cast INTEGER to VARCHAR
        WHERE c.client_id = $1;
    `;

    const { rows } = await db.query(query, [client_id]);

    if (rows.length === 0) return null;

    const client = rows[0];

    const transformedClient = {
      client_id: client.client_id,
      client_name: client.client_name,
      email: client.email,
      telephone: client.telephone,
      street: client.street,
      city: client.city,
      zipcode: client.zipcode,
      region: client.region_name || "Unknown", // Ensure region name is returned
      country: client.country_name || "Unknown", // Ensure country name is returned
      client_status: client.client_status , // Convert status
      created_at: client.created_at,
      role: client.role,
      profile_img: client.profile_img ? client.profile_img : null, // Return base64 directly
      address: client.address ? JSON.parse(client.address) : null,
    };

    await redis.setex(cacheKey, 3600, JSON.stringify(transformedClient));
    return transformedClient;
  }

  static async update({
    client_id,
    client_name,
    email,
    telephone,
    client_status,
    address,
    profile_img, // New field
  }) {
    if (!address || !address.street_1 || !address.city || !address.zip_code) {
      throw new Error("Complete address is required");
    }

    const street = address.street_1 + " " + (address.street_2 || "");
    const updates = [
      "client_name = $2",
      "email = $3",
      "telephone = $4",
      "address = $5",
      "street = $6",
      "city = $7",
      "zipcode = $8",
      "region = $9",
      "country = $10",
      "client_status = $11",
    ];
    const values = [
      client_id,
      client_name,
      email,
      telephone,
      JSON.stringify(address),
      street,
      address.city,
      address.zip_code,
      address.region,
      address.country,
      client_status,
    ];

    if (profile_img) {
      // Validate base64 format
      if (!/^data:image\/(png|jpg|jpeg);base64,/.test(profile_img)) {
        throw new Error("Invalid image format");
      }
      updates.push("profile_img = $" + (values.length + 1));
      values.push(profile_img);
    }

    const query = `
      UPDATE clients 
      SET ${updates.join(", ")}
      WHERE client_id = $1
      RETURNING client_id;
    `;

    try {
      const { rows } = await db.query(query, values);
      await redis.del(`clients:${client_id}`);
      await redis.del("clients:all");
      return rows[0].client_id;
    } catch (err) {
      console.error("❌ Error updating client:", err);
      throw err;
    }
  }

  static async delete(client_id) {
    await db.query("DELETE FROM clients WHERE client_id = $1", [client_id]);
    await redis.del(`clients:${client_id}`);
    await redis.del("clients:all");
    return { message: "Client deleted successfully" };
  }
}

module.exports = Vendor;
