const db = require("../../config/db");
const redis = require("../../config/redis");

class ClientInsta {
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS client_insta (
        client_id INT PRIMARY KEY,
        sales_channel_id INT NOT NULL REFERENCES master_sales_channel(channel_id) ON DELETE CASCADE,
        access_token TEXT NOT NULL,
        page_id TEXT NOT NULL,
        catalog_id TEXT NOT NULL,
        app_id TEXT NOT NULL,
        app_secret TEXT NOT NULL,
        username TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `;
    try {
      await db.query(query);
      console.log("✅ client_insta table is ready.");
    } catch (err) {
      console.error("❌ Error creating client_insta table:", err);
    }
  }

  static async create(data) {
    const query = `
      INSERT INTO client_insta (client_id, sales_channel_id, access_token, page_id, catalog_id, app_id, app_secret, username)
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
      RETURNING client_id;
    `;

    const values = [
      data.client_id,
      data.sales_channel_id,
      data.access_token,
      data.page_id,
      data.catalog_id,
      data.app_id,
      data.app_secret,
      data.username,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del("clients_insta:all");
      return rows[0].client_id;
    } catch (err) {
      console.error("❌ Error inserting Instagram client:", err);
      throw err;
    }
  }

  static async getAll() {
    const cachedClients = await redis.get("clients_insta:all");
    if (cachedClients) return JSON.parse(cachedClients);

    const { rows } = await db.query("SELECT * FROM client_insta");
    await redis.setex("clients_insta:all", 3600, JSON.stringify(rows));
    return rows;
  }

  static async getById(client_id) {
    const cachedClient = await redis.get(`clients_insta:${client_id}`);
    if (cachedClient) return JSON.parse(cachedClient);

    const {
      rows,
    } = await db.query("SELECT * FROM client_insta WHERE client_id = $1", [
      client_id,
    ]);
    if (rows.length === 0) return null;

    await redis.setex(
      `clients_insta:${client_id}`,
      3600,
      JSON.stringify(rows[0])
    );
    return rows[0];
  }

  static async delete(client_id) {
    await db.query("DELETE FROM client_insta WHERE client_id = $1", [
      client_id,
    ]);
    await redis.del(`clients_insta:${client_id}`);
    await redis.del("clients_insta:all");
    return { message: "Instagram client deleted successfully" };
  }
}

module.exports = ClientInsta;
