const db = require("../../config/db");
const redis = require("../../config/redis");

class ClientTwitter {
  // 🟢 Create client_twitter Table
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS client_twitter (
        client_id INT PRIMARY KEY,
        sales_channel_id INT NOT NULL REFERENCES master_sales_channel(channel_id) ON DELETE CASCADE,
        consumer_key TEXT NOT NULL,
        consumer_secret TEXT NOT NULL,
        access_token TEXT NOT NULL,
        access_token_secret TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `;

    try {
      await db.query(query);
      console.log("✅ client_twitter table is ready.");
    } catch (err) {
      console.error("❌ Error creating client_twitter table:", err);
    }
  }

  // 🟢 Create a new client record
  static async create(clientData) {
    const query = `
      INSERT INTO client_twitter (
        client_id, sales_channel_id, consumer_key, consumer_secret, access_token, access_token_secret
      ) VALUES ($1, $2, $3, $4, $5, $6)
      RETURNING client_id;
    `;

    const values = [
      clientData.client_id,
      clientData.sales_channel_id,
      clientData.consumer_key,
      clientData.consumer_secret,
      clientData.access_token,
      clientData.access_token_secret,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del("client_twitter:all");
      return rows[0].client_id;
    } catch (err) {
      console.error("❌ Error inserting client record:", err);
      throw err;
    }
  }

  // 🟢 Get all clients (with caching)
  static async getAll() {
    const cachedClients = await redis.get("client_twitter:all");
    if (cachedClients) {
      return JSON.parse(cachedClients);
    }

    try {
      const { rows } = await db.query("SELECT * FROM client_twitter");
      await redis.setex("client_twitter:all", 3600, JSON.stringify(rows));
      return rows;
    } catch (err) {
      console.error("❌ Error fetching client_twitter:", err);
      throw err;
    }
  }

  // 🟢 Get client by ID (with caching)
  static async getById(client_id) {
    const cachedClient = await redis.get(`client_twitter:${client_id}`);
    if (cachedClient) {
      return JSON.parse(cachedClient);
    }

    try {
      const {
        rows,
      } = await db.query("SELECT * FROM client_twitter WHERE client_id = $1", [
        client_id,
      ]);

      if (rows.length === 0) return null;

      await redis.setex(
        `client_twitter:${client_id}`,
        3600,
        JSON.stringify(rows[0])
      );
      return rows[0];
    } catch (err) {
      console.error("❌ Error fetching client by ID:", err);
      throw err;
    }
  }

  // 🟢 Update a client record
  static async update(clientData) {
    const query = `
      UPDATE client_twitter
      SET sales_channel_id = $2, consumer_key = $3, consumer_secret = $4, 
          access_token = $5, access_token_secret = $6
      WHERE client_id = $1
      RETURNING client_id;
    `;

    const values = [
      clientData.client_id,
      clientData.sales_channel_id,
      clientData.consumer_key,
      clientData.consumer_secret,
      clientData.access_token,
      clientData.access_token_secret,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del(`client_twitter:${clientData.client_id}`);
      await redis.del("client_twitter:all");
      return rows[0].client_id;
    } catch (err) {
      console.error("❌ Error updating client record:", err);
      throw err;
    }
  }

  // 🟢 Delete a client record
  static async delete(client_id) {
    try {
      await db.query("DELETE FROM client_twitter WHERE client_id = $1", [
        client_id,
      ]);
      await redis.del(`client_twitter:${client_id}`);
      await redis.del("client_twitter:all");
      return { message: "Client record deleted successfully" };
    } catch (err) {
      console.error("❌ Error deleting client record:", err);
      throw err;
    }
  }
}

module.exports = ClientTwitter;
