const db = require("../../config/db");
const redis = require("../../config/redis");

class MasterSalesChannel {
  // 🟢 Create master_sales_channel Table
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS master_sales_channel (
        channel_id SERIAL PRIMARY KEY,
        channel_name VARCHAR(255) NOT NULL UNIQUE,
        url TEXT NOT NULL UNIQUE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `;

    try {
      await db.query(query);
      console.log("✅ master_sales_channel table is ready.");
    } catch (err) {
      console.error("❌ Error creating master_sales_channel table:", err);
    }
  }

  // 🟢 Create a new sales channel
  static async create({ channel_name, url }) {
    const query = `
      INSERT INTO master_sales_channel (channel_name, url)
      VALUES ($1, $2)
      RETURNING channel_id;
    `;

    const values = [channel_name, url];

    try {
      const { rows } = await db.query(query, values);
      await redis.del("sales_channels:all"); // Invalidate cached list
      return rows[0].channel_id;
    } catch (err) {
      console.error("❌ Error inserting sales channel:", err);
      throw err;
    }
  }

  // 🟢 Get all sales channels (with caching)
  static async getAll() {
    const cachedChannels = await redis.get("sales_channels:all");
    if (cachedChannels) {
      return JSON.parse(cachedChannels);
    }

    const { rows } = await db.query("SELECT * FROM master_sales_channel");
    await redis.setex("sales_channels:all", 3600, JSON.stringify(rows)); // Cache for 1 hour
    return rows;
  }

  // 🟢 Get a sales channel by ID (with caching)
  static async getById(channel_id) {
    const cachedChannel = await redis.get(`sales_channels:${channel_id}`);
    if (cachedChannel) {
      return JSON.parse(cachedChannel);
    }

    const {
      rows,
    } = await db.query(
      "SELECT * FROM master_sales_channel WHERE channel_id = $1",
      [channel_id]
    );

    if (rows.length === 0) return null;

    await redis.setex(
      `sales_channels:${channel_id}`,
      3600,
      JSON.stringify(rows[0])
    );
    return rows[0];
  }

  // 🟢 Update a sales channel
  static async update({ channel_id, channel_name, url }) {
    const query = `
      UPDATE master_sales_channel
      SET channel_name = $2, url = $3
      WHERE channel_id = $1
      RETURNING channel_id;
    `;

    const values = [channel_id, channel_name, url];

    try {
      const { rows } = await db.query(query, values);
      await redis.del(`sales_channels:${channel_id}`);
      await redis.del("sales_channels:all");
      return rows[0].channel_id;
    } catch (err) {
      console.error("❌ Error updating sales channel:", err);
      throw err;
    }
  }

  // 🟢 Delete a sales channel
  static async delete(channel_id) {
    await db.query("DELETE FROM master_sales_channel WHERE channel_id = $1", [
      channel_id,
    ]);
    await redis.del(`sales_channels:${channel_id}`);
    await redis.del("sales_channels:all");
    return { message: "Sales channel deleted successfully" };
  }
}

module.exports = MasterSalesChannel;
