const express = require("express");
const Client = require("../models/client");
const auth = require("../middleware/auth");
const authMiddleware = require("../middleware/clientAuth").default;

const router = express.Router();
router.use(auth);
router.use(authMiddleware);
// Create a new vendor
router.post("/client/create", async (req, res, next) => {
  try {
    const {
      client_name,
      telephone,
      email,
      password,
      address,
      client_status,
      role,
      profile_img, // Optional
    } = req.body;

    if (!client_name)
      return res.status(400).json({ message: "client_name is required" });
    if (!email) return res.status(400).json({ message: "Email is required" });
    if (!telephone)
      return res.status(400).json({ message: "Telephone is required" });
    if (!password)
      return res.status(400).json({ message: "Password is required" });
    if (!address || !address.street_1 || !address.city || !address.zip_code) {
      return res.status(400).json({ message: "Complete address is required" });
    }

    // Validate Base64 Image Format only if profile_img is provided
    if (
      profile_img &&
      typeof profile_img === "string" &&
      !/^data:image\/(png|jpg|jpeg);base64,/.test(profile_img)
    ) {
      return res.status(400).json({ message: "Invalid image format" });
    }

    const clientData = {
      client_name,
      email,
      telephone,
      password,
      address,
      street: `${address.street_1} ${address.street_2 || ""}`.trim(),
      city: address.city,
      zipcode: address.zip_code,
      region: address.region,
      country: address.country,
      client_status,
      role: role || "client",
    };

    // Only include profile_img if provided
    if (profile_img) {
      clientData.profile_img = profile_img;
    }

    const client_id = await Client.create(clientData);

    res.status(201).json({
      status: true,
      message: "Client Created successfully",
      client_id,
    });
  } catch (err) {
    res
      .status(500)
      .json({ message: "Error creating client", error: err.message });
  }
});

// Get all vendors
router.get("/clients", async (req, res, next) => {
  try {
    const clients = await Client.getAll();
    res.json(clients);
  } catch (err) {
    next(err);
  }
});

// Get vendor by ID
router.get("/client/:client_id", auth, async (req, res, next) => {
  try {
    const client = await Client.getById(req.params.client_id);
    if (!client) return res.status(404).json({ message: "client not found" });
    res.json(client);
  } catch (err) {
    res
      .status(500)
      .json({ message: "Error fetching client", error: err.message });
  }
});

// Update vendor
router.put("/client/update", auth, async (req, res, next) => {
  try {
    const {
      client_id, // Fetching from req.body
      client_name,
      telephone,
      email,
      client_status,
      address,
      profile_img,
    } = req.body;

    if (!client_id)
      return res.status(400).json({ message: "client_id is required" });
    if (!client_name)
      return res.status(400).json({ message: "client_name is required" });
    if (!email) return res.status(400).json({ message: "Email is required" });
    if (!telephone)
      return res.status(400).json({ message: "Telephone is required" });
    if (!address || !address.street_1 || !address.city || !address.zip_code) {
      return res.status(400).json({ message: "Complete address is required" });
    }

    if (
      profile_img &&
      !/^data:image\/(png|jpg|jpeg);base64,/.test(profile_img)
    ) {
      return res.status(400).json({ message: "Invalid image format" });
    }

    const updatedClientId = await Client.update({
      client_id,
      client_name,
      email,
      telephone,
      client_status,
      address,
      profile_img, // Optional
    });

    res.json({
      status: true,
      message: "Client updated successfully",
      client_id: updatedClientId,
    });
  } catch (err) {
    res
      .status(500)
      .json({ message: "Error updating client", error: err.message });
  }
});

// Delete vendor
router.delete("/client/:client_id", auth, async (req, res) => {
  try {
    const result = await Client.delete(req.params.client_id);
    res.json(result);
  } catch (err) {
    res
      .status(500)
      .json({ message: "Error deleting Client", error: err.message });
  }
});

module.exports = router;
